#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>

#include "touch.h"

#ifndef UNITTEST
#include "input/mtouch_log.h"
#include "input/mtouch_driver.h"
#include "input/event_types.h"

#include "input/inputtrace.h"
#endif

void
cypress_dump_pcfg_data(cypress_dev_t* cypress)
{
    mtouch_info (cypress->log_name, "PCFG: Touch controller X-axis resolution: %d", cypress->register_map.pcfg_data.res_x);
    mtouch_info (cypress->log_name, "PCFG: Touch controller y-axis resolution: %d", cypress->register_map.pcfg_data.res_y);
    if (cypress->verbose >= 6) {
        mtouch_debug (cypress->log_name, "PCFG: Number of electrodes on x-axis of the panel: %d", cypress->register_map.pcfg_data.electrodes_x);
        mtouch_debug (cypress->log_name, "PCFG: Number of electrodes on y-axis of the panel: %d",cypress->register_map.pcfg_data.electrodes_y);
        mtouch_debug (cypress->log_name, "PCFG: X-axis length: 0x%04x", cypress->register_map.pcfg_data.len_x);
        mtouch_debug (cypress->log_name, "PCFG: y-axis length: 0x%04x", cypress->register_map.pcfg_data.len_y);
        mtouch_debug (cypress->log_name, "PCFG: Maximum Z value : 0x%04x", cypress->register_map.pcfg_data.max_z);
        mtouch_debug (cypress->log_name, "PCFG: Panel information register : %d", cypress->register_map.pcfg_data.panel_info0);
    }
}

void
cypress_dump_opmode_data(cypress_dev_t* cypress)
{
    mtouch_info (cypress->log_name, "OPINFO: Max touch points: %d", cypress->register_map.operating_mode_info->max_tchs);
    if (cypress->verbose >= 6) {
        mtouch_debug (cypress->log_name, "OPINFO: No of buttons configured: %d", cypress->register_map.operating_mode_info->num_btns);
        mtouch_debug (cypress->log_name, "OPINFO: Report size register: 0x%04x", cypress->register_map.operating_mode_info->rep_sz);
    }
}

void
cypress_dump_cypress_data(cypress_dev_t* cypress)
{
    mtouch_info (cypress->log_name, "SYSINFO: Cypress Controller on %s address 0x%x:", cypress->i2c_dev_name, cypress->i2c_slave_addr);
    mtouch_info (cypress->log_name, "SYSINFO: Product id: %d", cypress->register_map.cypress_data->ttpid);
    mtouch_info (cypress->log_name, "SYSINFO: Firmware major version: %d",cypress->register_map.cypress_data->fw_ver_major);
    mtouch_info (cypress->log_name, "SYSINFO: Firmware minor version: %d", cypress->register_map.cypress_data->fw_ver_minor);
    mtouch_info (cypress->log_name, "SYSINFO: Cypress revision control number: 0x%016llx", cypress->register_map.cypress_data->revctrl);
    mtouch_info (cypress->log_name, "SYSINFO: Bootloader major version: %d", cypress->register_map.cypress_data->bl_ver_major);
    mtouch_info (cypress->log_name, "SYSINFO: Bootloader minor version: %d", cypress->register_map.cypress_data->bl_ver_minor);
    mtouch_info (cypress->log_name, "SYSINFO: Family: Gen %d", cypress->silicon_gen);
    mtouch_info (cypress->log_name, "SYSINFO: silicon id: 0x%04x", cypress->register_map.cypress_data->si_id);
    mtouch_info (cypress->log_name, "SYSINFO: Manufacturing id size: %d", cypress->register_map.cypress_data->mfgid_sz);
    mtouch_info (cypress->log_name, "SYSINFO: CYITO version: 0x%04x", cypress->register_map.cypress_data->cyito_ver);
    mtouch_info (cypress->log_name, "SYSINFO: TTSP major version: %d", cypress->register_map.cypress_data->ttsp_ver_major);
    mtouch_info (cypress->log_name, "SYSINFO: TTSP minor version: %d", cypress->register_map.cypress_data->ttsp_ver_minor);
    mtouch_info (cypress->log_name, "SYSINFO: Device info: Endianness: %d", cypress->register_map.cypress_data->device_info);
}

void
cypress_dump_memory_map_data(cypress_dev_t* cypress)
{
    if (cypress->verbose >= 6) {
        mtouch_debug (cypress->log_name, "MEM_MAP: Host mode register: %d", cypress->register_map.sys_info_mem_map.hst_mode);
        mtouch_debug (cypress->log_name, "MEM_MAP: Reset detect register: %d",cypress->register_map.sys_info_mem_map.reset_detect);
        mtouch_debug (cypress->log_name, "MEM_MAP: Map size registor: %d", cypress->register_map.sys_info_mem_map.map_sz);
        mtouch_debug (cypress->log_name, "MEM_MAP: Cypress Data offset: %d", cypress->register_map.sys_info_mem_map.cydata_ofs);
        mtouch_debug (cypress->log_name, "MEM_MAP: Test offset: %d", cypress->register_map.sys_info_mem_map.test_ofs);
        mtouch_debug (cypress->log_name, "MEM_MAP: Panel Configuration ofset: %d", cypress->register_map.sys_info_mem_map.pcfg_ofs);
        mtouch_debug (cypress->log_name, "MEM_MAP: Operational Mode info offset: %d", cypress->register_map.sys_info_mem_map.opcfg_ofs);
        mtouch_debug (cypress->log_name, "MEM_MAP: Design Data offset: %d", cypress->register_map.sys_info_mem_map.ddata_ofs);
        mtouch_debug (cypress->log_name, "MEM_MAP: Manufacturer Data offset: %d", cypress->register_map.sys_info_mem_map.mdata_ofs);
    }
}

void
system_information_clean_up(cypress_dev_t* cypress)
{
    if (cypress->register_map.cypress_data) {
        free(cypress->register_map.cypress_data);
        cypress->register_map.cypress_data = NULL;
    }

    if (cypress->register_map.system_information_mode_reg) {
        free(cypress->register_map.system_information_mode_reg);
        cypress->register_map.system_information_mode_reg = NULL;
    }

    if (cypress->register_map.operating_mode_info) {
        free(cypress->register_map.operating_mode_info);
        cypress->register_map.operating_mode_info = NULL;
    }

    if (cypress->register_map.operating_mode_reg) {
        free(cypress->register_map.operating_mode_reg);
        cypress->register_map.operating_mode_reg = NULL;
    }

    if (cypress->register_map.customer_data.dsgn_data) {
        free (cypress->register_map.customer_data.dsgn_data);
        cypress->register_map.customer_data.dsgn_data = NULL;
    }

    if (cypress->register_map.customer_data.mfg_data) {
        free (cypress->register_map.customer_data.mfg_data);
        cypress->register_map.customer_data.mfg_data = NULL;
    }
}

int
cypress_read_pcfg_data(cypress_dev_t* cypress)
{
    uint8_t *reg;

    mtouch_info (cypress->log_name, "Fetching Panel Configuration Data");

    /* Read in the PCFG Data */
    reg =(uint8_t *) calloc(SYSTEM_INFORMATION_PCFG_SIZE, sizeof(uint8_t));
    if ((cypress->i2c_funcs.read_reg(cypress->i2c_fd, cypress->register_map.sys_info_mem_map.pcfg_ofs, SYSTEM_INFORMATION_PCFG_SIZE, reg)) == 0) {
        memcpy (&(cypress->register_map.pcfg_data), reg, SYSTEM_INFORMATION_PCFG_SIZE);

        cypress->register_map.pcfg_data.len_x = BSWAP_UINT16(cypress->register_map.pcfg_data.len_x);
        cypress->register_map.pcfg_data.len_y = BSWAP_UINT16(cypress->register_map.pcfg_data.len_y);
        cypress->register_map.pcfg_data.res_x = BSWAP_UINT16(cypress->register_map.pcfg_data.res_x);
        cypress->register_map.pcfg_data.res_y = BSWAP_UINT16(cypress->register_map.pcfg_data.res_y);
        cypress->register_map.pcfg_data.max_z = BSWAP_UINT16(cypress->register_map.pcfg_data.max_z);
    } else {
        mtouch_error (cypress->log_name, "Failed to read Panel Configuration Data");
        return -1;
    }

    if (cypress->verbose) {
        mtouch_info (cypress->log_name, "Screen origin is: %s %s", (cypress->register_map.pcfg_data.res_x & 0x8000) ? "Bottom" : "Top",
                                                                 (cypress->register_map.pcfg_data.res_x & 0x8000) ? "right" : "left");
    }

    if (cypress->verbose)
        cypress_dump_pcfg_data(cypress);
    free(reg);
    return EOK;
}

int
cypress_read_customer_data(cypress_dev_t* cypress)
{
    uint8_t *reg;
    char *buf;
    int design_data_size = (cypress->register_map.sys_info_mem_map.mdata_ofs - cypress->register_map.sys_info_mem_map.ddata_ofs);
    int manufacturer_data_size = (cypress->register_map.sys_info_mem_map.map_sz - cypress->register_map.sys_info_mem_map.mdata_ofs);

    mtouch_info (cypress->log_name, "Fetching Design and Manufacturer Data");

    /* Read in the Design data */
    reg =(uint8_t *) calloc(design_data_size, sizeof(uint8_t));
    if ((cypress->i2c_funcs.read_reg(cypress->i2c_fd, cypress->register_map.sys_info_mem_map.ddata_ofs, design_data_size, reg)) == 0) {
        cypress->register_map.customer_data.dsgn_data =(uint8_t *) calloc(design_data_size, sizeof(uint8_t));

        if (cypress->register_map.customer_data.dsgn_data == NULL) {
            mtouch_error (cypress->log_name, "Failed to calloc space for Design Data");
            return -1;
        }

        memcpy (cypress->register_map.customer_data.dsgn_data, reg, design_data_size);
        buf = (char *) malloc (design_data_size);
        if (buf == NULL) {
            mtouch_error (cypress->log_name, "CUST: Failed to allocate buffer space to dump design data");
        } else {
            sprintf (buf, "%u ", *(cypress->register_map.customer_data.dsgn_data));
            if (cypress->verbose >= 6)
                mtouch_debug (cypress->log_name, "CUST: Design specific data: %s, len=%d", buf, design_data_size);
            free (buf);
        }
    } else {
        mtouch_error (cypress->log_name, "Failed to read in Design Data (size %d)", design_data_size);
    }
    free(reg);
    /* Read in the Manufacturer data */
    reg =(uint8_t *) calloc(manufacturer_data_size, sizeof(uint8_t));
    if ((cypress->i2c_funcs.read_reg(cypress->i2c_fd, cypress->register_map.sys_info_mem_map.mdata_ofs, manufacturer_data_size, reg)) == 0) {
        cypress->register_map.customer_data.mfg_data =(uint8_t *) calloc(manufacturer_data_size, sizeof(uint8_t));

        if (cypress->register_map.customer_data.mfg_data == NULL) {
            mtouch_error (cypress->log_name, "Failed to calloc space for Manufacturer data");
            return -1;
        }

        memcpy (cypress->register_map.customer_data.mfg_data, reg, manufacturer_data_size);
        buf = (char *) malloc (manufacturer_data_size);
        if (buf == NULL) {
            mtouch_error (cypress->log_name, "CUST: Failed to allocate buffer space to dump manufacture data");
        } else {
            sprintf (buf, "%u ", *(cypress->register_map.customer_data.mfg_data));
            if (cypress->verbose >= 6)
                mtouch_debug (cypress->log_name, "CUST: Manufacturer specific data: %s, len =%d", buf, manufacturer_data_size);
            free (buf);
        }
    }else {
        mtouch_error (cypress->log_name, "Failed to read in Manufacturer Data (size %d)", manufacturer_data_size);
    }

    free(reg);
    return EOK;
}

int
cypress_read_cypress_data(cypress_dev_t* cypress)
{
    uint8_t reg;
    uint8_t *buf;
    uint8_t cypress_data_size, mfgid_size;

    if (cypress->verbose > 4)
        mtouch_info (cypress->log_name, "Reading Cypress Controller Data");


    /* Read in size of manufacturing ID byte at MFGID_SZ offset */
    if ((cypress->i2c_funcs.read_reg(cypress->i2c_fd, cypress->register_map.sys_info_mem_map.cydata_ofs + MFGID_SZ, sizeof(uint8_t), &reg)) == 0) {
        mfgid_size = reg;
        if (mfgid_size > MAX_MFGID_SZ) {
            mtouch_error (cypress->log_name, "actual MFGID size is large than MAX_MFGID_SZ");
            return -1;
        }
        cypress_data_size = (mfgid_size + SYSTEM_INFORMATION_CYPRESS_DATA_SIZE);

        if (cypress->register_map.cypress_data != NULL) {
            mtouch_warn (cypress->log_name, "cypress_data is allocated before release it");
            free(cypress->register_map.cypress_data);
        }

        cypress->register_map.cypress_data = (cypress_data_t *)calloc(1, sizeof(*cypress->register_map.cypress_data));

        if (cypress->register_map.cypress_data == NULL) {
            mtouch_error (cypress->log_name, "Failed to calloc space for system info map");
            return -1;
        }

        /* Read in all of the Cypress data and copy it to the struct */
        buf =(uint8_t *) calloc(cypress_data_size, sizeof(uint8_t));
        if ((cypress->i2c_funcs.read_reg(cypress->i2c_fd, cypress->register_map.sys_info_mem_map.cydata_ofs, cypress_data_size, buf)) == 0) {
            /* copy from TTPIDH to MFGID_SZ, 19 bytes in total */
            memcpy(cypress->register_map.cypress_data, buf, MFGID_SZ + 1);
            /* copy manufracture id string */
            memcpy(cypress->register_map.cypress_data->mfgid, &buf[MFGID], mfgid_size);

            /* copy rest of 7 registers after manufracture id string */
            memcpy(cypress->register_map.cypress_data->reserved2, &buf[mfgid_size + MFGID], SYSTEM_INFORMATION_CYPRESS_DATA_SIZE - MFGID_SZ - 6);

            cypress->register_map.cypress_data->ttpid = BSWAP_UINT16(cypress->register_map.cypress_data->ttpid);
            cypress->register_map.cypress_data->si_id = BSWAP_UINT16(cypress->register_map.cypress_data->si_id);
            cypress->register_map.cypress_data->cyito_ver = BSWAP_UINT16(cypress->register_map.cypress_data->cyito_ver);
            cypress->register_map.cypress_data->revctrl = BSWAP_UINT64(cypress->register_map.cypress_data->revctrl);

            if (cypress->verbose)
                cypress_dump_cypress_data(cypress);

        } else {
            mtouch_error (cypress->log_name, "Failed to read cypress data");
            return -1;
        }

    } else {
        mtouch_error (cypress->log_name, "Failed to read MFGID_SZ register");
        return -1;
    }

    /* Read in Panel Configuration Data */
    if (EOK != cypress_read_pcfg_data(cypress)) {
        return -1;
    }

    /* Grab the Customer (Design and Manufacturer Data) */
    if (EOK != cypress_read_customer_data(cypress)) {
        return -1;
    }

    return EOK;
}


int
cypress_map_operating_mode_registers(cypress_dev_t* cypress)
{
    uint8_t *reg = 0;

    if (cypress->verbose > 4)
        mtouch_info (cypress->log_name, "Reading Operating Mode offsets");

    cypress->opcfg_size = cypress->register_map.sys_info_mem_map.ddata_ofs - cypress->register_map.sys_info_mem_map.opcfg_ofs;

    if (cypress->opcfg_size <= 0) {
        mtouch_error (cypress->log_name, "Invalid opcfg size %d", cypress->opcfg_size);
        return -1;
    }

    /* Read in system information reg data for operating mode reg */
    reg =(uint8_t *) calloc(cypress->opcfg_size, sizeof(uint8_t));
    if (reg == NULL) {
            mtouch_error (cypress->log_name, "Failed to calloc space for reg");
            return -1;
    }
    if ((cypress->i2c_funcs.read_reg(cypress->i2c_fd, cypress->register_map.sys_info_mem_map.opcfg_ofs, cypress->opcfg_size, reg)) == 0) {

        if (cypress->register_map.operating_mode_info != NULL) {
            mtouch_warn (cypress->log_name, "operation_mode_info is allocated before release it");
            free(cypress->register_map.operating_mode_info);
        }
        /* Copy the contents of reg to the struct so we can use it later */
        cypress->register_map.operating_mode_info =(operating_mode_info_t *) calloc(cypress->opcfg_size, sizeof(uint8_t));

        if (cypress->register_map.operating_mode_info == NULL) {
            mtouch_error (cypress->log_name, "Failed to calloc space for system info map");
            return -1;
        }

        memcpy (cypress->register_map.operating_mode_info, reg, cypress->opcfg_size);

        cypress->register_map.operating_mode_info->rep_sz = BSWAP_UINT16(cypress->register_map.operating_mode_info->rep_sz);

    } else {
        mtouch_error (cypress->log_name, "Failed to read opcfg data, i2c failed");
        return -1;
    }

    if (cypress->register_map.operating_mode_info->num_btns) {

        /* Calculate how many regsiters we need to read from as there are 4 buttons for each BUTTONX register */
        cypress->num_button_regs = ((cypress->register_map.operating_mode_info->num_btns / CYPRESS_BUTTONS_PER_REG) +
                                    ((cypress->register_map.operating_mode_info->num_btns % CYPRESS_BUTTONS_PER_REG) ? 1 : 0));
    }

    /* check if rtd record size is configured */
    if ((cypress->register_map.operating_mode_info->rtd_rec_ofs != 0) &&
        (cypress->opcfg_size > RTD_REC_OFS) &&
        (cypress->register_map.operating_mode_info->rtd_rec_siz != 0)) {

        mtouch_info(cypress->log_name, "RTD funtionality is enabled ");
        mtouch_info (cypress->log_name, "OPINFO: RTD record size: %d", cypress->register_map.operating_mode_info->rtd_rec_siz);
        cypress->rtd_enable = 1;                    /* set this flag to send rtd messages to applications */
        cypress->rtd_len = MAX_RTD_DATA_LENGTH;
    } else {
        /* if rtd rec size is set to 0 or not configured, then RTD function is disabled */
        mtouch_info(cypress->log_name, "RTD funtionality is not enabled in the chip ");
    }

    if (cypress->verbose)
        cypress_dump_opmode_data(cypress);

    return EOK;
}

int
cypress_read_memory_map(cypress_dev_t* cypress)
{
    uint8_t *reg = 0;

    reg =(uint8_t *) calloc(SYSTEM_INFORMATION_INITIAL_MAP_SIZE, sizeof(uint8_t));
    if ((cypress->i2c_funcs.read_reg(cypress->i2c_fd, 0, SYSTEM_INFORMATION_INITIAL_MAP_SIZE, reg)) == 0) {
        cypress->register_map.sys_info_mem_map.hst_mode = reg[HST_MODE];
        cypress->register_map.sys_info_mem_map.reset_detect = reg[RESET_DETECT];
        cypress->register_map.sys_info_mem_map.map_sz = (reg[MAP_SZH] << 8);
        cypress->register_map.sys_info_mem_map.map_sz |= reg[MAP_SZL];
        cypress->register_map.sys_info_mem_map.cydata_ofs = (reg[CYDATA_OFSH] << 8);
        cypress->register_map.sys_info_mem_map.cydata_ofs |= reg[CYDATA_OFSL];
        cypress->register_map.sys_info_mem_map.pcfg_ofs = (reg[PCFG_OFSH] << 8);
        cypress->register_map.sys_info_mem_map.pcfg_ofs |= reg[PCFG_OFSL];
        cypress->register_map.sys_info_mem_map.opcfg_ofs = (reg[OPCFG_OFSH] << 8);
        cypress->register_map.sys_info_mem_map.opcfg_ofs |= reg[OPCFG_OFSL];
        cypress->register_map.sys_info_mem_map.ddata_ofs = (reg[DDATA_OFSH] << 8);
        cypress->register_map.sys_info_mem_map.ddata_ofs |= reg[DDATA_OFSL];
        cypress->register_map.sys_info_mem_map.mdata_ofs = (reg[MDATA_OFSH] << 8);
        cypress->register_map.sys_info_mem_map.mdata_ofs |= reg[MDATA_OFSL];

        /* Read Cypress Data */
        if (cypress_read_cypress_data(cypress) != EOK) {
            goto fail1;
        }

        /* Get Operating Mode Register Map */
        if (cypress_map_operating_mode_registers(cypress) != EOK) {
            goto fail1;
        }
    } else {
        mtouch_error (cypress->log_name, "Failed to read the system information memory map size");
        goto fail2;
    }
    if (cypress->verbose)
        cypress_dump_memory_map_data(cypress);

    return EOK;

fail1:
    system_information_clean_up(cypress);
fail2:
    return -1;
}


#if defined(__QNXNTO__) && defined(__USESRCVERSION)
#include <sys/srcversion.h>
__SRCVERSION("$URL: http://svn.ott.qnx.com/product/branches/7.0.0/trunk/hardware/mtouch/cypress/system_information.c $ $Rev: 879933 $")
#endif
